'use client'

import AcmeLogo from '@/app/ui/acme-logo';
import { ArrowRightIcon } from '@heroicons/react/24/outline';
import Link from 'next/link';
import styles from '@/app/ui/home.module.css';
import { lusitana } from '@/app/ui/fonts';
import Image from 'next/image';
import { ChangeEvent, useEffect, useState } from 'react';
import { getAllVideos } from '@/services/video';
import { Video } from '@/types/Video';
import { Button, Input } from '@headlessui/react';
import { Dialog, DialogBackdrop, DialogPanel, DialogTitle } from '@headlessui/react';
import { useRouter } from 'next/navigation';
import { initiateMpesaC2b, queryPayment } from '@/services/payment';

type IState = { 
  video: Video | null; 
  videos: Video[]; 
  showBuyModal: boolean; 
  paymentInitiated: boolean;
  paymentProcessing: boolean;
  paymentProcessed?: boolean;
  paymentStatus?: string;
  paymentMessage?: string;
  reference?: string;
  paymentMethod?: string;
};

export default function Page() {
  const router = useRouter();
  const [state, setState] = useState<IState>({
    video: null,
    videos: [],
    showBuyModal: false,
    paymentInitiated: false,
    paymentProcessing: false,
  });

  const [formData, setFormData] = useState({
    phone: ""
  });

  useEffect(() => {
    getAllVideos().then((res) => {
      setState({
        ...state,
        videos: res,
      })
    })
  }, [])

  const onBuy = (video: Video) => {
    setState({
      ...state,
      video,
      showBuyModal: true,
    })
  }

  const handleBuy = () => {
    if (state.video?.id && formData.phone) {
      setState({
        ...state,
        paymentInitiated: true
      })

      initiateMpesaC2b({
        video_id: state.video?.id,
        phone: formData.phone
      }).then((res: { ref: string, status: string, message: string }) => {
        setState({
          ...state,
          paymentInitiated: false,
          paymentProcessing: true,
          reference: res.ref,
          paymentMethod: 'mpesa',
          paymentStatus: res.status,
          paymentMessage: res.message
        })
      }).catch(err => {
        console.log('err -->', err)
      })
    }
  }

  const checkStatus = () => {
    if(state?.reference && state.paymentMethod) {
      queryPayment({
        ref: state.reference,
        payment_method: state.paymentMethod
      }).then(res => {
        console.log('res -->', res)
        setState({
          ...state,
          paymentProcessed: true,
          paymentStatus: res.status,
          paymentMessage: res.message
        })

        // Redirect user
        if(res.status === 'Paid') {
          setTimeout(() => {
            router.push(`payment?ref=${state.reference}`)
          }, 1000)
        }
      })
    }
  }

  const handleChange = (e: ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData((prevState) => ({ ...prevState, [name]: value }));
  };

  const handleRetryPayment = () => {
    setState({
      ...state,
      paymentInitiated: false,
      paymentProcessing: false,
      paymentProcessed: false,
    })
  }

  const handleCloseModal = () => {
    setState({
      ...state,
      showBuyModal: false,
      paymentInitiated: false,
      paymentProcessing: false,
      paymentProcessed: false,
    })
  }

  return (
    <main className="flex min-h-screen flex-col p-6">
      <div className="flex h-20 shrink-0 items-end rounded-lg bg-blue-500 p-4 md:h-30">
        <AcmeLogo />
      </div>
      <div className="mt-4 flex grow flex-col gap-4 md:flex-row">
        <div className="flex flex-col justify-center gap-6 rounded-lg bg-gray-50 px-6 py-10 md:w-2/5 md:px-20">
          <div
            className={styles.shape}
          />
          <p className={`text-xl text-gray-800 md:text-3xl md:leading-normal ${lusitana.className} antialiased`}>
            <strong>Welcome to Acme.</strong> This is the example for the{' '}
            <a href="https://nextjs.org/learn/" className="text-blue-500">
              Next.js Learn Course
            </a>
            , brought to you by Vercel.
          </p>
          <Link
            href="/login"
            className="flex items-center gap-5 self-start rounded-lg bg-blue-500 px-6 py-3 text-sm font-medium text-white transition-colors hover:bg-blue-400 md:text-base"
          >
            <span>Log in</span> <ArrowRightIcon className="w-5 md:w-6" />
          </Link>
        </div>
        <div className="flex items-center justify-center p-6 md:w-3/5 md:px-28 md:py-12">
          {/* Add Hero Images Here */}
          <Image
            src="/hero-desktop.png"
            width={1000}
            height={760}
            className="hidden md:block"
            alt="Screenshots of the dashboard project showing desktop version"
          />
          <Image
            src="/hero-mobile.png"
            width={560}
            height={620}
            className="block md:hidden"
            alt="Screenshots of the dashboard project showing desktop version"
          />
        </div>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mt-3 md:mt-12">
        {state.videos.map((video) => (
          <div key={video.id} className="max-w-sm bg-white border border-gray-200 rounded-lg shadow dark:bg-gray-800 dark:border-gray-700">
              <a href="#">
                  <img className="rounded-t-lg" src={video.poster ?? 'https://images.unsplash.com/photo-1445985543470-41fba5c3144a?q=80&w=1470&auto=format&fit=crop&ixlib=rb-4.0.3&ixid=M3wxMjA3fDB8MHxwaG90by1wYWdlfHx8fGVufDB8fHx8fA%3D%3D'} alt="" />
              </a>
              <div className="p-5">
                  <a href="#">
                      <h5 className="mb-2 text-1xl font-bold tracking-tight text-gray-900 dark:text-white">{video.name}</h5>
                  </a>
                  <p className="mb-3 font-normal text-gray-700 dark:text-gray-400">{video.description}</p>
                  <Button className="px-6 py-2 rounded text-white bg-blue-500" onClick={() => onBuy(video)}>Buy</Button>
              </div>
          </div>

        ))}
      </div>

      <Dialog open={state.showBuyModal} onClose={handleCloseModal}>
        <DialogBackdrop className="fixed inset-0 bg-black/30" />
        <div className="fixed inset-0 z-10 w-screen overflow-y-auto">
          <div className="flex min-h-full items-center justify-center p-4 relative">

            <DialogPanel
              transition
              className="w-full max-w-md rounded-xl bg-white relative p-6  duration-300 ease-out data-[closed]:transform-[scale(95%)] data-[closed]:opacity-0"
            >
              <Button id="closeModal" className="absolute top-2 right-2 text-xl text-gray-600 hover:text-gray-900" onClick={handleCloseModal}>
                &times;
              </Button>

              <DialogTitle className="text-lg font-medium">
                Purchase {state.video?.name} @ KES {state.video?.price}
              </DialogTitle>
              {!state.paymentProcessing && <>
              <Input placeholder="Your Phone Number" className="mt-3 block w-full rounded-lg border py-1.5 px-3 text-sm/6" name="phone" value={formData.phone} onChange={handleChange} />
              <Button className="mt-3 px-6 py-2 rounded text-white bg-blue-500 flex items-center" onClick={handleBuy}>
                {state.paymentInitiated && <svg className="animate-spin -ml-1 mr-2 h-5 w-5 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                  <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                  <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                </svg>} <span>Buy</span></Button></>
              }

              {(state.paymentProcessing && !state.paymentProcessed) && <div className="mt-3 text-center">
                <p>Please enter your PIN on your phone to complete this payment. Click complete after you have entered your PIN.</p>
                <div className="flex items-center mt-5 justify-center">
                  <Button className="mr-4 px-6 py-2 rounded text-white bg-blue-500 flex items-center" onClick={checkStatus}>
                    <span>Complete</span>
                  </Button>
                  <Button className="py-2 px-3 rounded bg-gray-300" onClick={() => setState({...state, paymentProcessing: false})}>
                    Cancel
                  </Button>
                </div>
                </div>
              }

              {state.paymentProcessed && <div className="mt-5 text-center">
                  <p>{state.paymentMessage}</p>
                  {state.paymentStatus === 'Failed' && <Button className="mx-auto mt-5 px-6 py-2 rounded text-white bg-blue-500 flex items-center" onClick={handleRetryPayment}>
                    <span>Retry</span>
                  </Button>}
                </div>
              }
            </DialogPanel>
          </div>
        </div>
      </Dialog>
      <footer className="mt-4"></footer>
    </main>
  );
}
