import Link from "next/link"
import { ChangeEvent, useEffect, useRef, useState } from "react"
import {
    Dialog, DialogBackdrop, DialogPanel,
    Disclosure, DisclosureButton, DisclosurePanel, Description,
    Field, Input, Label, Textarea
} from '@headlessui/react'
import { ChevronDownIcon } from '@heroicons/react/20/solid'
import { ArrowUpTrayIcon } from "@heroicons/react/24/outline"
import clsx from 'clsx'
import Cropper, { ReactCropperElement } from "react-cropper";

import Uploader from "@/components/uploader"
import ShakaPlayer from "@/components/shaka-player";
import { Button } from "@/app/ui/button"
import { getVideo, updateVideo } from "@/services/video"
import { Video } from "@/types/Video"

import "cropperjs/dist/cropper.css";

const VideoDetail = (props: { id: number }) => {
    const { id } = props

    const [video, setVideo] = useState<Video>()
    const [cropVideo, setCropVideo] = useState(false)
    const [replaceVideo, setReplaceVideo] = useState(false)

    const [formData, setFormData] = useState<Video>({
        name: "",
        description: "",
        price: 0,
        poster: null
    })

    const [uploadUrl, setUploadUrl] = useState<string>("")

    const videoThumbnail = video?.poster ?? "//shaka-player-demo.appspot.com/assets/poster.jpg";
    const mpdFile = video?.video_url

    const cropperRef = useRef<ReactCropperElement>(null);

    const onCrop = () => {
        const cropper = cropperRef.current?.cropper;

        const croppedData = cropper?.getCroppedCanvas().toDataURL();

        if (croppedData) {
            setFormData((prevState) => ({ ...prevState, poster: croppedData }));
        }
    };

    const handleAfterCrop = () => {
        setCropVideo(false) // Hide modal
        handleSave()
    }

    useEffect(() => {
        getVideo(+id).then(res => setVideo(res.video))
    }, [id])

    useEffect(() => {
        if (video) {
            setFormData({
                name: video.name,
                description: video.description,
                price: video.price,
                poster: video.poster
            })
        }
    }, [video])

    const handleOnUpload = () => {
        setReplaceVideo(false)
        getVideo(+id).then(res => setVideo(res.video))
    }

    const handleSave = () => {
        if (video?.id) {
            updateVideo(video.id, formData).then(() => {
                getVideo(+id).then(res => setVideo(res.video))
            })
        }
    }

    const handleChange = (e: ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
        const { name, value } = e.target;

        if (name === 'poster' && value) {
            const target = e.target as HTMLInputElement;
            const file = target.files?.[0];
            if (file) {
                const reader = new FileReader();
                reader.readAsDataURL(file);
                reader.onloadend = () => {
                    setCropVideo(true)
                    setUploadUrl(reader.result as string)
                };
            }
        } else {
            setFormData((prevState) => ({ ...prevState, [name]: value }));
        }
    };

    return (
        <>
            <div className="rounded border border-stroke bg-white px-3 pb-2.5 pt-6 shadow dark:border-strokedark dark:bg-boxdark sm:px-7.5 xl:pb-1">
                <div className="flex justify-between">
                    <h4 className="mb-6 text-xl font-semibold text-black dark:text-white">
                        {video?.name}
                    </h4>

                    <div className="flex gap-4">
                        <Link href={"/dashboard/videos"}>
                            <button className="bg-gray-300 rounded px-3 py-2">Back</button>
                        </Link>
                    </div>
                </div>
                <hr className="mb-4 h-0.5 border-t-0 bg-neutral-100 dark:bg-white/10" />
                <div className="grid grid-cols-[60%_40%] gap-4">
                    <ShakaPlayer
                        manifestUri={mpdFile}
                        posterUrl={videoThumbnail}
                    />
                    <div>
                        <Disclosure as="div" className="p-6 border rounded mb-3">
                            <DisclosureButton className="group flex w-full items-center justify-between size-4">
                                <span className="text-sm/6 font-medium">
                                    Replace Video
                                </span>
                                <ChevronDownIcon className="size-5 fill-white/60 group-data-[hover]:fill-white/50 group-data-[open]:rotate-180" />
                            </DisclosureButton>
                            <DisclosurePanel className="mt-2 text-sm/5">
                                <Button onClick={() => setReplaceVideo(true)}><ArrowUpTrayIcon className="w-4 mr-2" /> Select from the device</Button>
                            </DisclosurePanel>
                        </Disclosure>

                        <Disclosure as="div" className="p-6 border rounded mb-3">
                            <DisclosureButton className="group flex w-full items-center justify-between">
                                <span className="text-sm/6 font-medium">
                                    Details
                                </span>
                                <ChevronDownIcon className="size-5 group-data-[open]:rotate-180" />
                            </DisclosureButton>
                            <DisclosurePanel className="mt-2 text-sm/5">

                                <Field>
                                    <Label className="text-sm/6 font-medium text-white">Name</Label>
                                    <Description className="text-sm/6">Video name</Description>
                                    <Input
                                        className={clsx(
                                            'mt-3 block w-full rounded-lg border  py-1.5 px-3 text-sm/6',
                                            'focus:outline-none data-[focus]:outline-2 data-[focus]:-outline-offset-2 data-[focus]:outline-white/25'
                                        )}
                                        name="name"
                                        value={formData?.name}
                                        onChange={handleChange}
                                    />
                                </Field>

                                <Field>
                                    <Label className="text-sm/6 font-medium">Price</Label>
                                    <Input
                                        className={clsx(
                                            'mt-3 block w-full rounded-lg border  py-1.5 px-3 text-sm/6',
                                            'focus:outline-none data-[focus]:outline-2 data-[focus]:-outline-offset-2 data-[focus]:outline-white/25'
                                        )}
                                        name="price"
                                        value={formData?.price}
                                        onChange={handleChange}
                                    />
                                </Field>

                                <Field>
                                    <Label className="text-sm/6 font-medium">Description</Label>
                                    <Textarea
                                        className={clsx(
                                            'mt-3 block w-full resize-none rounded-lg border py-1.5 px-3 text-sm/6',
                                            'focus:outline-none data-[focus]:outline-2 data-[focus]:-outline-offset-2 data-[focus]:outline-white/25'
                                        )}
                                        rows={3}
                                        name="description"
                                        value={formData?.description}
                                        onChange={handleChange}
                                    />
                                </Field>
                                <Button className="mt-3" onClick={handleSave}>Save</Button>
                            </DisclosurePanel>
                        </Disclosure>

                        <Disclosure as="div" className="p-6 border rounded">
                            <DisclosureButton className="group flex w-full items-center justify-between">
                                <span className="text-sm/6 font-medium">
                                    Change Poster
                                </span>
                                <ChevronDownIcon className="size-5 group-data-[open]:rotate-180" />
                            </DisclosureButton>
                            <DisclosurePanel className="mt-2 text-sm/5">
                                {video?.poster && <img src={video?.poster} className="mb-3" />}

                                <input
                                    className="block w-full text-sm text-gray-900 border border-gray-300 rounded-lg cursor-pointer bg-gray-50 dark:text-gray-400 focus:outline-none dark:bg-gray-700 dark:border-gray-600 dark:placeholder-gray-400"
                                    id="file_input"
                                    type="file"
                                    name="poster"
                                    accept="image/*"
                                    onChange={handleChange}
                                />
                            </DisclosurePanel>
                        </Disclosure>
                    </div>
                </div>
            </div>

            <Dialog open={replaceVideo} as="div" className="relative z-10 focus:outline-none" onClose={() => setReplaceVideo(false)}>
                <DialogBackdrop className="fixed inset-0 bg-black/30" />
                <div className="fixed inset-0 z-10 w-screen overflow-y-auto">
                    <div className="flex min-h-full items-center justify-center p-4">
                        <DialogPanel
                            transition
                            className="w-full max-w-md rounded-xl bg-white relative p-6  duration-300 ease-out data-[closed]:transform-[scale(95%)] data-[closed]:opacity-0"
                        >
                            <button id="closeModal" className={`absolute top-2 right-2 bottom-4 text-gray-600 hover:text-gray-900`} onClick={() => setReplaceVideo(false)}>
                                &times;
                            </button>
                            <Uploader onUpload={handleOnUpload} />
                        </DialogPanel>
                    </div>
                </div>
            </Dialog>

            <Dialog open={cropVideo} as="div" className="relative z-10 focus:outline-none" onClose={() => setCropVideo(false)}>
                <DialogBackdrop className="fixed inset-0 bg-black/30" />
                <div className="fixed inset-0 z-10 w-screen overflow-y-auto">
                    <div className="flex min-h-full items-center justify-center p-4">
                        <DialogPanel
                            transition
                            className="w-full max-w-md rounded-xl bg-white relative px-6  duration-300 ease-out data-[closed]:transform-[scale(95%)] data-[closed]:opacity-0"
                        >
                            <div className="w-full mb-8 relative">
                                <button id="closeModal" className={`absolute top-2 right-2 bottom-4 text-gray-600 hover:text-gray-900`} onClick={() => setCropVideo(false)}>
                                    &times;
                                </button>
                            </div>
                            <div className="mt-8">
                                <Cropper
                                    src={uploadUrl}
                                    style={{ height: 400, width: "100%" }}
                                    // Cropper.js options
                                    initialAspectRatio={16 / 9}
                                    aspectRatio={16 / 9}
                                    guides={false}
                                    crop={onCrop}
                                    ref={cropperRef}
                                />
                            </div>
                            <div className="mt-4 mb-4">
                                <Button
                                    className="inline-flex items-center gap-2 rounded-md bg-gray-700 py-1.5 px-3 text-sm/6 font-semibold text-white shadow-inner shadow-white/10 focus:outline-none data-[hover]:bg-gray-600 data-[focus]:outline-1 data-[focus]:outline-white data-[open]:bg-gray-700"
                                    onClick={handleAfterCrop}
                                >
                                    Crop
                                </Button>
                            </div>
                        </DialogPanel>
                    </div>
                </div>
            </Dialog>
        </>
    )
}

export default VideoDetail